%% @author Bob Ippolito <bob@mochimedia.com>
%% @copyright 2007 Mochi Media, Inc.
%%
%% Permission is hereby granted, free of charge, to any person obtaining a
%% copy of this software and associated documentation files (the "Software"),
%% to deal in the Software without restriction, including without limitation
%% the rights to use, copy, modify, merge, publish, distribute, sublicense,
%% and/or sell copies of the Software, and to permit persons to whom the
%% Software is furnished to do so, subject to the following conditions:
%%
%% The above copyright notice and this permission notice shall be included in
%% all copies or substantial portions of the Software.
%%
%% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
%% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
%% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
%% THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
%% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
%% FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
%% DEALINGS IN THE SOFTWARE.

%% @doc Converts HTML 5 charrefs and entities to codepoints (or lists of code points).
-module(mochiweb_charref).
-export([charref/1]).

%% External API.

%% @doc Convert a decimal charref, hex charref, or html entity to a unicode
%%      codepoint, or return undefined on failure.
%%      The input should not include an ampersand or semicolon.
%%      charref("#38") = 38, charref("#x26") = 38, charref("amp") = 38.
-spec charref(binary() | string()) -> integer() | [integer()] | undefined.
charref(B) when is_binary(B) ->
    charref(binary_to_list(B));
charref([$#, C | L]) when C =:= $x orelse C =:= $X ->
    try erlang:list_to_integer(L, 16)
    catch
        error:badarg -> undefined
    end;
charref([$# | L]) ->
    try list_to_integer(L)
    catch
        error:badarg -> undefined
    end;
charref(L) ->
    entity(L).

%% Internal API.

%% [2011-10-14] Generated from:
%% http://www.w3.org/TR/html5/named-character-references.html

entity("AElig") -> 16#000C6;
entity("AMP") -> 16#00026;
entity("Aacute") -> 16#000C1;
entity("Abreve") -> 16#00102;
entity("Acirc") -> 16#000C2;
entity("Acy") -> 16#00410;
entity("Afr") -> 16#1D504;
entity("Agrave") -> 16#000C0;
entity("Alpha") -> 16#00391;
entity("Amacr") -> 16#00100;
entity("And") -> 16#02A53;
entity("Aogon") -> 16#00104;
entity("Aopf") -> 16#1D538;
entity("ApplyFunction") -> 16#02061;
entity("Aring") -> 16#000C5;
entity("Ascr") -> 16#1D49C;
entity("Assign") -> 16#02254;
entity("Atilde") -> 16#000C3;
entity("Auml") -> 16#000C4;
entity("Backslash") -> 16#02216;
entity("Barv") -> 16#02AE7;
entity("Barwed") -> 16#02306;
entity("Bcy") -> 16#00411;
entity("Because") -> 16#02235;
entity("Bernoullis") -> 16#0212C;
entity("Beta") -> 16#00392;
entity("Bfr") -> 16#1D505;
entity("Bopf") -> 16#1D539;
entity("Breve") -> 16#002D8;
entity("Bscr") -> 16#0212C;
entity("Bumpeq") -> 16#0224E;
entity("CHcy") -> 16#00427;
entity("COPY") -> 16#000A9;
entity("Cacute") -> 16#00106;
entity("Cap") -> 16#022D2;
entity("CapitalDifferentialD") -> 16#02145;
entity("Cayleys") -> 16#0212D;
entity("Ccaron") -> 16#0010C;
entity("Ccedil") -> 16#000C7;
entity("Ccirc") -> 16#00108;
entity("Cconint") -> 16#02230;
entity("Cdot") -> 16#0010A;
entity("Cedilla") -> 16#000B8;
entity("CenterDot") -> 16#000B7;
entity("Cfr") -> 16#0212D;
entity("Chi") -> 16#003A7;
entity("CircleDot") -> 16#02299;
entity("CircleMinus") -> 16#02296;
entity("CirclePlus") -> 16#02295;
entity("CircleTimes") -> 16#02297;
entity("ClockwiseContourIntegral") -> 16#02232;
entity("CloseCurlyDoubleQuote") -> 16#0201D;
entity("CloseCurlyQuote") -> 16#02019;
entity("Colon") -> 16#02237;
entity("Colone") -> 16#02A74;
entity("Congruent") -> 16#02261;
entity("Conint") -> 16#0222F;
entity("ContourIntegral") -> 16#0222E;
entity("Copf") -> 16#02102;
entity("Coproduct") -> 16#02210;
entity("CounterClockwiseContourIntegral") -> 16#02233;
entity("Cross") -> 16#02A2F;
entity("Cscr") -> 16#1D49E;
entity("Cup") -> 16#022D3;
entity("CupCap") -> 16#0224D;
entity("DD") -> 16#02145;
entity("DDotrahd") -> 16#02911;
entity("DJcy") -> 16#00402;
entity("DScy") -> 16#00405;
entity("DZcy") -> 16#0040F;
entity("Dagger") -> 16#02021;
entity("Darr") -> 16#021A1;
entity("Dashv") -> 16#02AE4;
entity("Dcaron") -> 16#0010E;
entity("Dcy") -> 16#00414;
entity("Del") -> 16#02207;
entity("Delta") -> 16#00394;
entity("Dfr") -> 16#1D507;
entity("DiacriticalAcute") -> 16#000B4;
entity("DiacriticalDot") -> 16#002D9;
entity("DiacriticalDoubleAcute") -> 16#002DD;
entity("DiacriticalGrave") -> 16#00060;
entity("DiacriticalTilde") -> 16#002DC;
entity("Diamond") -> 16#022C4;
entity("DifferentialD") -> 16#02146;
entity("Dopf") -> 16#1D53B;
entity("Dot") -> 16#000A8;
entity("DotDot") -> 16#020DC;
entity("DotEqual") -> 16#02250;
entity("DoubleContourIntegral") -> 16#0222F;
entity("DoubleDot") -> 16#000A8;
entity("DoubleDownArrow") -> 16#021D3;
entity("DoubleLeftArrow") -> 16#021D0;
entity("DoubleLeftRightArrow") -> 16#021D4;
entity("DoubleLeftTee") -> 16#02AE4;
entity("DoubleLongLeftArrow") -> 16#027F8;
entity("DoubleLongLeftRightArrow") -> 16#027FA;
entity("DoubleLongRightArrow") -> 16#027F9;
entity("DoubleRightArrow") -> 16#021D2;
entity("DoubleRightTee") -> 16#022A8;
entity("DoubleUpArrow") -> 16#021D1;
entity("DoubleUpDownArrow") -> 16#021D5;
entity("DoubleVerticalBar") -> 16#02225;
entity("DownArrow") -> 16#02193;
entity("DownArrowBar") -> 16#02913;
entity("DownArrowUpArrow") -> 16#021F5;
entity("DownBreve") -> 16#00311;
entity("DownLeftRightVector") -> 16#02950;
entity("DownLeftTeeVector") -> 16#0295E;
entity("DownLeftVector") -> 16#021BD;
entity("DownLeftVectorBar") -> 16#02956;
entity("DownRightTeeVector") -> 16#0295F;
entity("DownRightVector") -> 16#021C1;
entity("DownRightVectorBar") -> 16#02957;
entity("DownTee") -> 16#022A4;
entity("DownTeeArrow") -> 16#021A7;
entity("Downarrow") -> 16#021D3;
entity("Dscr") -> 16#1D49F;
entity("Dstrok") -> 16#00110;
entity("ENG") -> 16#0014A;
entity("ETH") -> 16#000D0;
entity("Eacute") -> 16#000C9;
entity("Ecaron") -> 16#0011A;
entity("Ecirc") -> 16#000CA;
entity("Ecy") -> 16#0042D;
entity("Edot") -> 16#00116;
entity("Efr") -> 16#1D508;
entity("Egrave") -> 16#000C8;
entity("Element") -> 16#02208;
entity("Emacr") -> 16#00112;
entity("EmptySmallSquare") -> 16#025FB;
entity("EmptyVerySmallSquare") -> 16#025AB;
entity("Eogon") -> 16#00118;
entity("Eopf") -> 16#1D53C;
entity("Epsilon") -> 16#00395;
entity("Equal") -> 16#02A75;
entity("EqualTilde") -> 16#02242;
entity("Equilibrium") -> 16#021CC;
entity("Escr") -> 16#02130;
entity("Esim") -> 16#02A73;
entity("Eta") -> 16#00397;
entity("Euml") -> 16#000CB;
entity("Exists") -> 16#02203;
entity("ExponentialE") -> 16#02147;
entity("Fcy") -> 16#00424;
entity("Ffr") -> 16#1D509;
entity("FilledSmallSquare") -> 16#025FC;
entity("FilledVerySmallSquare") -> 16#025AA;
entity("Fopf") -> 16#1D53D;
entity("ForAll") -> 16#02200;
entity("Fouriertrf") -> 16#02131;
entity("Fscr") -> 16#02131;
entity("GJcy") -> 16#00403;
entity("GT") -> 16#0003E;
entity("Gamma") -> 16#00393;
entity("Gammad") -> 16#003DC;
entity("Gbreve") -> 16#0011E;
entity("Gcedil") -> 16#00122;
entity("Gcirc") -> 16#0011C;
entity("Gcy") -> 16#00413;
entity("Gdot") -> 16#00120;
entity("Gfr") -> 16#1D50A;
entity("Gg") -> 16#022D9;
entity("Gopf") -> 16#1D53E;
entity("GreaterEqual") -> 16#02265;
entity("GreaterEqualLess") -> 16#022DB;
entity("GreaterFullEqual") -> 16#02267;
entity("GreaterGreater") -> 16#02AA2;
entity("GreaterLess") -> 16#02277;
entity("GreaterSlantEqual") -> 16#02A7E;
entity("GreaterTilde") -> 16#02273;
entity("Gscr") -> 16#1D4A2;
entity("Gt") -> 16#0226B;
entity("HARDcy") -> 16#0042A;
entity("Hacek") -> 16#002C7;
entity("Hat") -> 16#0005E;
entity("Hcirc") -> 16#00124;
entity("Hfr") -> 16#0210C;
entity("HilbertSpace") -> 16#0210B;
entity("Hopf") -> 16#0210D;
entity("HorizontalLine") -> 16#02500;
entity("Hscr") -> 16#0210B;
entity("Hstrok") -> 16#00126;
entity("HumpDownHump") -> 16#0224E;
entity("HumpEqual") -> 16#0224F;
entity("IEcy") -> 16#00415;
entity("IJlig") -> 16#00132;
entity("IOcy") -> 16#00401;
entity("Iacute") -> 16#000CD;
entity("Icirc") -> 16#000CE;
entity("Icy") -> 16#00418;
entity("Idot") -> 16#00130;
entity("Ifr") -> 16#02111;
entity("Igrave") -> 16#000CC;
entity("Im") -> 16#02111;
entity("Imacr") -> 16#0012A;
entity("ImaginaryI") -> 16#02148;
entity("Implies") -> 16#021D2;
entity("Int") -> 16#0222C;
entity("Integral") -> 16#0222B;
entity("Intersection") -> 16#022C2;
entity("InvisibleComma") -> 16#02063;
entity("InvisibleTimes") -> 16#02062;
entity("Iogon") -> 16#0012E;
entity("Iopf") -> 16#1D540;
entity("Iota") -> 16#00399;
entity("Iscr") -> 16#02110;
entity("Itilde") -> 16#00128;
entity("Iukcy") -> 16#00406;
entity("Iuml") -> 16#000CF;
entity("Jcirc") -> 16#00134;
entity("Jcy") -> 16#00419;
entity("Jfr") -> 16#1D50D;
entity("Jopf") -> 16#1D541;
entity("Jscr") -> 16#1D4A5;
entity("Jsercy") -> 16#00408;
entity("Jukcy") -> 16#00404;
entity("KHcy") -> 16#00425;
entity("KJcy") -> 16#0040C;
entity("Kappa") -> 16#0039A;
entity("Kcedil") -> 16#00136;
entity("Kcy") -> 16#0041A;
entity("Kfr") -> 16#1D50E;
entity("Kopf") -> 16#1D542;
entity("Kscr") -> 16#1D4A6;
entity("LJcy") -> 16#00409;
entity("LT") -> 16#0003C;
entity("Lacute") -> 16#00139;
entity("Lambda") -> 16#0039B;
entity("Lang") -> 16#027EA;
entity("Laplacetrf") -> 16#02112;
entity("Larr") -> 16#0219E;
entity("Lcaron") -> 16#0013D;
entity("Lcedil") -> 16#0013B;
entity("Lcy") -> 16#0041B;
entity("LeftAngleBracket") -> 16#027E8;
entity("LeftArrow") -> 16#02190;
entity("LeftArrowBar") -> 16#021E4;
entity("LeftArrowRightArrow") -> 16#021C6;
entity("LeftCeiling") -> 16#02308;
entity("LeftDoubleBracket") -> 16#027E6;
entity("LeftDownTeeVector") -> 16#02961;
entity("LeftDownVector") -> 16#021C3;
entity("LeftDownVectorBar") -> 16#02959;
entity("LeftFloor") -> 16#0230A;
entity("LeftRightArrow") -> 16#02194;
entity("LeftRightVector") -> 16#0294E;
entity("LeftTee") -> 16#022A3;
entity("LeftTeeArrow") -> 16#021A4;
entity("LeftTeeVector") -> 16#0295A;
entity("LeftTriangle") -> 16#022B2;
entity("LeftTriangleBar") -> 16#029CF;
entity("LeftTriangleEqual") -> 16#022B4;
entity("LeftUpDownVector") -> 16#02951;
entity("LeftUpTeeVector") -> 16#02960;
entity("LeftUpVector") -> 16#021BF;
entity("LeftUpVectorBar") -> 16#02958;
entity("LeftVector") -> 16#021BC;
entity("LeftVectorBar") -> 16#02952;
entity("Leftarrow") -> 16#021D0;
entity("Leftrightarrow") -> 16#021D4;
entity("LessEqualGreater") -> 16#022DA;
entity("LessFullEqual") -> 16#02266;
entity("LessGreater") -> 16#02276;
entity("LessLess") -> 16#02AA1;
entity("LessSlantEqual") -> 16#02A7D;
entity("LessTilde") -> 16#02272;
entity("Lfr") -> 16#1D50F;
entity("Ll") -> 16#022D8;
entity("Lleftarrow") -> 16#021DA;
entity("Lmidot") -> 16#0013F;
entity("LongLeftArrow") -> 16#027F5;
entity("LongLeftRightArrow") -> 16#027F7;
entity("LongRightArrow") -> 16#027F6;
entity("Longleftarrow") -> 16#027F8;
entity("Longleftrightarrow") -> 16#027FA;
entity("Longrightarrow") -> 16#027F9;
entity("Lopf") -> 16#1D543;
entity("LowerLeftArrow") -> 16#02199;
entity("LowerRightArrow") -> 16#02198;
entity("Lscr") -> 16#02112;
entity("Lsh") -> 16#021B0;
entity("Lstrok") -> 16#00141;
entity("Lt") -> 16#0226A;
entity("Map") -> 16#02905;
entity("Mcy") -> 16#0041C;
entity("MediumSpace") -> 16#0205F;
entity("Mellintrf") -> 16#02133;
entity("Mfr") -> 16#1D510;
entity("MinusPlus") -> 16#02213;
entity("Mopf") -> 16#1D544;
entity("Mscr") -> 16#02133;
entity("Mu") -> 16#0039C;
entity("NJcy") -> 16#0040A;
entity("Nacute") -> 16#00143;
entity("Ncaron") -> 16#00147;
entity("Ncedil") -> 16#00145;
entity("Ncy") -> 16#0041D;
entity("NegativeMediumSpace") -> 16#0200B;
entity("NegativeThickSpace") -> 16#0200B;
entity("NegativeThinSpace") -> 16#0200B;
entity("NegativeVeryThinSpace") -> 16#0200B;
entity("NestedGreaterGreater") -> 16#0226B;
entity("NestedLessLess") -> 16#0226A;
entity("NewLine") -> 16#0000A;
entity("Nfr") -> 16#1D511;
entity("NoBreak") -> 16#02060;
entity("NonBreakingSpace") -> 16#000A0;
entity("Nopf") -> 16#02115;
entity("Not") -> 16#02AEC;
entity("NotCongruent") -> 16#02262;
entity("NotCupCap") -> 16#0226D;
entity("NotDoubleVerticalBar") -> 16#02226;
entity("NotElement") -> 16#02209;
entity("NotEqual") -> 16#02260;
entity("NotEqualTilde") -> [16#02242, 16#00338];
entity("NotExists") -> 16#02204;
entity("NotGreater") -> 16#0226F;
entity("NotGreaterEqual") -> 16#02271;
entity("NotGreaterFullEqual") -> [16#02267, 16#00338];
entity("NotGreaterGreater") -> [16#0226B, 16#00338];
entity("NotGreaterLess") -> 16#02279;
entity("NotGreaterSlantEqual") -> [16#02A7E, 16#00338];
entity("NotGreaterTilde") -> 16#02275;
entity("NotHumpDownHump") -> [16#0224E, 16#00338];
entity("NotHumpEqual") -> [16#0224F, 16#00338];
entity("NotLeftTriangle") -> 16#022EA;
entity("NotLeftTriangleBar") -> [16#029CF, 16#00338];
entity("NotLeftTriangleEqual") -> 16#022EC;
entity("NotLess") -> 16#0226E;
entity("NotLessEqual") -> 16#02270;
entity("NotLessGreater") -> 16#02278;
entity("NotLessLess") -> [16#0226A, 16#00338];
entity("NotLessSlantEqual") -> [16#02A7D, 16#00338];
entity("NotLessTilde") -> 16#02274;
entity("NotNestedGreaterGreater") -> [16#02AA2, 16#00338];
entity("NotNestedLessLess") -> [16#02AA1, 16#00338];
entity("NotPrecedes") -> 16#02280;
entity("NotPrecedesEqual") -> [16#02AAF, 16#00338];
entity("NotPrecedesSlantEqual") -> 16#022E0;
entity("NotReverseElement") -> 16#0220C;
entity("NotRightTriangle") -> 16#022EB;
entity("NotRightTriangleBar") -> [16#029D0, 16#00338];
entity("NotRightTriangleEqual") -> 16#022ED;
entity("NotSquareSubset") -> [16#0228F, 16#00338];
entity("NotSquareSubsetEqual") -> 16#022E2;
entity("NotSquareSuperset") -> [16#02290, 16#00338];
entity("NotSquareSupersetEqual") -> 16#022E3;
entity("NotSubset") -> [16#02282, 16#020D2];
entity("NotSubsetEqual") -> 16#02288;
entity("NotSucceeds") -> 16#02281;
entity("NotSucceedsEqual") -> [16#02AB0, 16#00338];
entity("NotSucceedsSlantEqual") -> 16#022E1;
entity("NotSucceedsTilde") -> [16#0227F, 16#00338];
entity("NotSuperset") -> [16#02283, 16#020D2];
entity("NotSupersetEqual") -> 16#02289;
entity("NotTilde") -> 16#02241;
entity("NotTildeEqual") -> 16#02244;
entity("NotTildeFullEqual") -> 16#02247;
entity("NotTildeTilde") -> 16#02249;
entity("NotVerticalBar") -> 16#02224;
entity("Nscr") -> 16#1D4A9;
entity("Ntilde") -> 16#000D1;
entity("Nu") -> 16#0039D;
entity("OElig") -> 16#00152;
entity("Oacute") -> 16#000D3;
entity("Ocirc") -> 16#000D4;
entity("Ocy") -> 16#0041E;
entity("Odblac") -> 16#00150;
entity("Ofr") -> 16#1D512;
entity("Ograve") -> 16#000D2;
entity("Omacr") -> 16#0014C;
entity("Omega") -> 16#003A9;
entity("Omicron") -> 16#0039F;
entity("Oopf") -> 16#1D546;
entity("OpenCurlyDoubleQuote") -> 16#0201C;
entity("OpenCurlyQuote") -> 16#02018;
entity("Or") -> 16#02A54;
entity("Oscr") -> 16#1D4AA;
entity("Oslash") -> 16#000D8;
entity("Otilde") -> 16#000D5;
entity("Otimes") -> 16#02A37;
entity("Ouml") -> 16#000D6;
entity("OverBar") -> 16#0203E;
entity("OverBrace") -> 16#023DE;
entity("OverBracket") -> 16#023B4;
entity("OverParenthesis") -> 16#023DC;
entity("PartialD") -> 16#02202;
entity("Pcy") -> 16#0041F;
entity("Pfr") -> 16#1D513;
entity("Phi") -> 16#003A6;
entity("Pi") -> 16#003A0;
entity("PlusMinus") -> 16#000B1;
entity("Poincareplane") -> 16#0210C;
entity("Popf") -> 16#02119;
entity("Pr") -> 16#02ABB;
entity("Precedes") -> 16#0227A;
entity("PrecedesEqual") -> 16#02AAF;
entity("PrecedesSlantEqual") -> 16#0227C;
entity("PrecedesTilde") -> 16#0227E;
entity("Prime") -> 16#02033;
entity("Product") -> 16#0220F;
entity("Proportion") -> 16#02237;
entity("Proportional") -> 16#0221D;
entity("Pscr") -> 16#1D4AB;
entity("Psi") -> 16#003A8;
entity("QUOT") -> 16#00022;
entity("Qfr") -> 16#1D514;
entity("Qopf") -> 16#0211A;
entity("Qscr") -> 16#1D4AC;
entity("RBarr") -> 16#02910;
entity("REG") -> 16#000AE;
entity("Racute") -> 16#00154;
entity("Rang") -> 16#027EB;
entity("Rarr") -> 16#021A0;
entity("Rarrtl") -> 16#02916;
entity("Rcaron") -> 16#00158;
entity("Rcedil") -> 16#00156;
entity("Rcy") -> 16#00420;
entity("Re") -> 16#0211C;
entity("ReverseElement") -> 16#0220B;
entity("ReverseEquilibrium") -> 16#021CB;
entity("ReverseUpEquilibrium") -> 16#0296F;
entity("Rfr") -> 16#0211C;
entity("Rho") -> 16#003A1;
entity("RightAngleBracket") -> 16#027E9;
entity("RightArrow") -> 16#02192;
entity("RightArrowBar") -> 16#021E5;
entity("RightArrowLeftArrow") -> 16#021C4;
entity("RightCeiling") -> 16#02309;
entity("RightDoubleBracket") -> 16#027E7;
entity("RightDownTeeVector") -> 16#0295D;
entity("RightDownVector") -> 16#021C2;
entity("RightDownVectorBar") -> 16#02955;
entity("RightFloor") -> 16#0230B;
entity("RightTee") -> 16#022A2;
entity("RightTeeArrow") -> 16#021A6;
entity("RightTeeVector") -> 16#0295B;
entity("RightTriangle") -> 16#022B3;
entity("RightTriangleBar") -> 16#029D0;
entity("RightTriangleEqual") -> 16#022B5;
entity("RightUpDownVector") -> 16#0294F;
entity("RightUpTeeVector") -> 16#0295C;
entity("RightUpVector") -> 16#021BE;
entity("RightUpVectorBar") -> 16#02954;
entity("RightVector") -> 16#021C0;
entity("RightVectorBar") -> 16#02953;
entity("Rightarrow") -> 16#021D2;
entity("Ropf") -> 16#0211D;
entity("RoundImplies") -> 16#02970;
entity("Rrightarrow") -> 16#021DB;
entity("Rscr") -> 16#0211B;
entity("Rsh") -> 16#021B1;
entity("RuleDelayed") -> 16#029F4;
entity("SHCHcy") -> 16#00429;
entity("SHcy") -> 16#00428;
entity("SOFTcy") -> 16#0042C;
entity("Sacute") -> 16#0015A;
entity("Sc") -> 16#02ABC;
entity("Scaron") -> 16#00160;
entity("Scedil") -> 16#0015E;
entity("Scirc") -> 16#0015C;
entity("Scy") -> 16#00421;
entity("Sfr") -> 16#1D516;
entity("ShortDownArrow") -> 16#02193;
entity("ShortLeftArrow") -> 16#02190;
entity("ShortRightArrow") -> 16#02192;
entity("ShortUpArrow") -> 16#02191;
entity("Sigma") -> 16#003A3;
entity("SmallCircle") -> 16#02218;
entity("Sopf") -> 16#1D54A;
entity("Sqrt") -> 16#0221A;
entity("Square") -> 16#025A1;
entity("SquareIntersection") -> 16#02293;
entity("SquareSubset") -> 16#0228F;
entity("SquareSubsetEqual") -> 16#02291;
entity("SquareSuperset") -> 16#02290;
entity("SquareSupersetEqual") -> 16#02292;
entity("SquareUnion") -> 16#02294;
entity("Sscr") -> 16#1D4AE;
entity("Star") -> 16#022C6;
entity("Sub") -> 16#022D0;
entity("Subset") -> 16#022D0;
entity("SubsetEqual") -> 16#02286;
entity("Succeeds") -> 16#0227B;
entity("SucceedsEqual") -> 16#02AB0;
entity("SucceedsSlantEqual") -> 16#0227D;
entity("SucceedsTilde") -> 16#0227F;
entity("SuchThat") -> 16#0220B;
entity("Sum") -> 16#02211;
entity("Sup") -> 16#022D1;
entity("Superset") -> 16#02283;
entity("SupersetEqual") -> 16#02287;
entity("Supset") -> 16#022D1;
entity("THORN") -> 16#000DE;
entity("TRADE") -> 16#02122;
entity("TSHcy") -> 16#0040B;
entity("TScy") -> 16#00426;
entity("Tab") -> 16#00009;
entity("Tau") -> 16#003A4;
entity("Tcaron") -> 16#00164;
entity("Tcedil") -> 16#00162;
entity("Tcy") -> 16#00422;
entity("Tfr") -> 16#1D517;
entity("Therefore") -> 16#02234;
entity("Theta") -> 16#00398;
entity("ThickSpace") -> [16#0205F, 16#0200A];
entity("ThinSpace") -> 16#02009;
entity("Tilde") -> 16#0223C;
entity("TildeEqual") -> 16#02243;
entity("TildeFullEqual") -> 16#02245;
entity("TildeTilde") -> 16#02248;
entity("Topf") -> 16#1D54B;
entity("TripleDot") -> 16#020DB;
entity("Tscr") -> 16#1D4AF;
entity("Tstrok") -> 16#00166;
entity("Uacute") -> 16#000DA;
entity("Uarr") -> 16#0219F;
entity("Uarrocir") -> 16#02949;
entity("Ubrcy") -> 16#0040E;
entity("Ubreve") -> 16#0016C;
entity("Ucirc") -> 16#000DB;
entity("Ucy") -> 16#00423;
entity("Udblac") -> 16#00170;
entity("Ufr") -> 16#1D518;
entity("Ugrave") -> 16#000D9;
entity("Umacr") -> 16#0016A;
entity("UnderBar") -> 16#0005F;
entity("UnderBrace") -> 16#023DF;
entity("UnderBracket") -> 16#023B5;
entity("UnderParenthesis") -> 16#023DD;
entity("Union") -> 16#022C3;
entity("UnionPlus") -> 16#0228E;
entity("Uogon") -> 16#00172;
entity("Uopf") -> 16#1D54C;
entity("UpArrow") -> 16#02191;
entity("UpArrowBar") -> 16#02912;
entity("UpArrowDownArrow") -> 16#021C5;
entity("UpDownArrow") -> 16#02195;
entity("UpEquilibrium") -> 16#0296E;
entity("UpTee") -> 16#022A5;
entity("UpTeeArrow") -> 16#021A5;
entity("Uparrow") -> 16#021D1;
entity("Updownarrow") -> 16#021D5;
entity("UpperLeftArrow") -> 16#02196;
entity("UpperRightArrow") -> 16#02197;
entity("Upsi") -> 16#003D2;
entity("Upsilon") -> 16#003A5;
entity("Uring") -> 16#0016E;
entity("Uscr") -> 16#1D4B0;
entity("Utilde") -> 16#00168;
entity("Uuml") -> 16#000DC;
entity("VDash") -> 16#022AB;
entity("Vbar") -> 16#02AEB;
entity("Vcy") -> 16#00412;
entity("Vdash") -> 16#022A9;
entity("Vdashl") -> 16#02AE6;
entity("Vee") -> 16#022C1;
entity("Verbar") -> 16#02016;
entity("Vert") -> 16#02016;
entity("VerticalBar") -> 16#02223;
entity("VerticalLine") -> 16#0007C;
entity("VerticalSeparator") -> 16#02758;
entity("VerticalTilde") -> 16#02240;
entity("VeryThinSpace") -> 16#0200A;
entity("Vfr") -> 16#1D519;
entity("Vopf") -> 16#1D54D;
entity("Vscr") -> 16#1D4B1;
entity("Vvdash") -> 16#022AA;
entity("Wcirc") -> 16#00174;
entity("Wedge") -> 16#022C0;
entity("Wfr") -> 16#1D51A;
entity("Wopf") -> 16#1D54E;
entity("Wscr") -> 16#1D4B2;
entity("Xfr") -> 16#1D51B;
entity("Xi") -> 16#0039E;
entity("Xopf") -> 16#1D54F;
entity("Xscr") -> 16#1D4B3;
entity("YAcy") -> 16#0042F;
entity("YIcy") -> 16#00407;
entity("YUcy") -> 16#0042E;
entity("Yacute") -> 16#000DD;
entity("Ycirc") -> 16#00176;
entity("Ycy") -> 16#0042B;
entity("Yfr") -> 16#1D51C;
entity("Yopf") -> 16#1D550;
entity("Yscr") -> 16#1D4B4;
entity("Yuml") -> 16#00178;
entity("ZHcy") -> 16#00416;
entity("Zacute") -> 16#00179;
entity("Zcaron") -> 16#0017D;
entity("Zcy") -> 16#00417;
entity("Zdot") -> 16#0017B;
entity("ZeroWidthSpace") -> 16#0200B;
entity("Zeta") -> 16#00396;
entity("Zfr") -> 16#02128;
entity("Zopf") -> 16#02124;
entity("Zscr") -> 16#1D4B5;
entity("aacute") -> 16#000E1;
entity("abreve") -> 16#00103;
entity("ac") -> 16#0223E;
entity("acE") -> [16#0223E, 16#00333];
entity("acd") -> 16#0223F;
entity("acirc") -> 16#000E2;
entity("acute") -> 16#000B4;
entity("acy") -> 16#00430;
entity("aelig") -> 16#000E6;
entity("af") -> 16#02061;
entity("afr") -> 16#1D51E;
entity("agrave") -> 16#000E0;
entity("alefsym") -> 16#02135;
entity("aleph") -> 16#02135;
entity("alpha") -> 16#003B1;
entity("amacr") -> 16#00101;
entity("amalg") -> 16#02A3F;
entity("amp") -> 16#00026;
entity("and") -> 16#02227;
entity("andand") -> 16#02A55;
entity("andd") -> 16#02A5C;
entity("andslope") -> 16#02A58;
entity("andv") -> 16#02A5A;
entity("ang") -> 16#02220;
entity("ange") -> 16#029A4;
entity("angle") -> 16#02220;
entity("angmsd") -> 16#02221;
entity("angmsdaa") -> 16#029A8;
entity("angmsdab") -> 16#029A9;
entity("angmsdac") -> 16#029AA;
entity("angmsdad") -> 16#029AB;
entity("angmsdae") -> 16#029AC;
entity("angmsdaf") -> 16#029AD;
entity("angmsdag") -> 16#029AE;
entity("angmsdah") -> 16#029AF;
entity("angrt") -> 16#0221F;
entity("angrtvb") -> 16#022BE;
entity("angrtvbd") -> 16#0299D;
entity("angsph") -> 16#02222;
entity("angst") -> 16#000C5;
entity("angzarr") -> 16#0237C;
entity("aogon") -> 16#00105;
entity("aopf") -> 16#1D552;
entity("ap") -> 16#02248;
entity("apE") -> 16#02A70;
entity("apacir") -> 16#02A6F;
entity("ape") -> 16#0224A;
entity("apid") -> 16#0224B;
entity("apos") -> 16#00027;
entity("approx") -> 16#02248;
entity("approxeq") -> 16#0224A;
entity("aring") -> 16#000E5;
entity("ascr") -> 16#1D4B6;
entity("ast") -> 16#0002A;
entity("asymp") -> 16#02248;
entity("asympeq") -> 16#0224D;
entity("atilde") -> 16#000E3;
entity("auml") -> 16#000E4;
entity("awconint") -> 16#02233;
entity("awint") -> 16#02A11;
entity("bNot") -> 16#02AED;
entity("backcong") -> 16#0224C;
entity("backepsilon") -> 16#003F6;
entity("backprime") -> 16#02035;
entity("backsim") -> 16#0223D;
entity("backsimeq") -> 16#022CD;
entity("barvee") -> 16#022BD;
entity("barwed") -> 16#02305;
entity("barwedge") -> 16#02305;
entity("bbrk") -> 16#023B5;
entity("bbrktbrk") -> 16#023B6;
entity("bcong") -> 16#0224C;
entity("bcy") -> 16#00431;
entity("bdquo") -> 16#0201E;
entity("becaus") -> 16#02235;
entity("because") -> 16#02235;
entity("bemptyv") -> 16#029B0;
entity("bepsi") -> 16#003F6;
entity("bernou") -> 16#0212C;
entity("beta") -> 16#003B2;
entity("beth") -> 16#02136;
entity("between") -> 16#0226C;
entity("bfr") -> 16#1D51F;
entity("bigcap") -> 16#022C2;
entity("bigcirc") -> 16#025EF;
entity("bigcup") -> 16#022C3;
entity("bigodot") -> 16#02A00;
entity("bigoplus") -> 16#02A01;
entity("bigotimes") -> 16#02A02;
entity("bigsqcup") -> 16#02A06;
entity("bigstar") -> 16#02605;
entity("bigtriangledown") -> 16#025BD;
entity("bigtriangleup") -> 16#025B3;
entity("biguplus") -> 16#02A04;
entity("bigvee") -> 16#022C1;
entity("bigwedge") -> 16#022C0;
entity("bkarow") -> 16#0290D;
entity("blacklozenge") -> 16#029EB;
entity("blacksquare") -> 16#025AA;
entity("blacktriangle") -> 16#025B4;
entity("blacktriangledown") -> 16#025BE;
entity("blacktriangleleft") -> 16#025C2;
entity("blacktriangleright") -> 16#025B8;
entity("blank") -> 16#02423;
entity("blk12") -> 16#02592;
entity("blk14") -> 16#02591;
entity("blk34") -> 16#02593;
entity("block") -> 16#02588;
entity("bne") -> [16#0003D, 16#020E5];
entity("bnequiv") -> [16#02261, 16#020E5];
entity("bnot") -> 16#02310;
entity("bopf") -> 16#1D553;
entity("bot") -> 16#022A5;
entity("bottom") -> 16#022A5;
entity("bowtie") -> 16#022C8;
entity("boxDL") -> 16#02557;
entity("boxDR") -> 16#02554;
entity("boxDl") -> 16#02556;
entity("boxDr") -> 16#02553;
entity("boxH") -> 16#02550;
entity("boxHD") -> 16#02566;
entity("boxHU") -> 16#02569;
entity("boxHd") -> 16#02564;
entity("boxHu") -> 16#02567;
entity("boxUL") -> 16#0255D;
entity("boxUR") -> 16#0255A;
entity("boxUl") -> 16#0255C;
entity("boxUr") -> 16#02559;
entity("boxV") -> 16#02551;
entity("boxVH") -> 16#0256C;
entity("boxVL") -> 16#02563;
entity("boxVR") -> 16#02560;
entity("boxVh") -> 16#0256B;
entity("boxVl") -> 16#02562;
entity("boxVr") -> 16#0255F;
entity("boxbox") -> 16#029C9;
entity("boxdL") -> 16#02555;
entity("boxdR") -> 16#02552;
entity("boxdl") -> 16#02510;
entity("boxdr") -> 16#0250C;
entity("boxh") -> 16#02500;
entity("boxhD") -> 16#02565;
entity("boxhU") -> 16#02568;
entity("boxhd") -> 16#0252C;
entity("boxhu") -> 16#02534;
entity("boxminus") -> 16#0229F;
entity("boxplus") -> 16#0229E;
entity("boxtimes") -> 16#022A0;
entity("boxuL") -> 16#0255B;
entity("boxuR") -> 16#02558;
entity("boxul") -> 16#02518;
entity("boxur") -> 16#02514;
entity("boxv") -> 16#02502;
entity("boxvH") -> 16#0256A;
entity("boxvL") -> 16#02561;
entity("boxvR") -> 16#0255E;
entity("boxvh") -> 16#0253C;
entity("boxvl") -> 16#02524;
entity("boxvr") -> 16#0251C;
entity("bprime") -> 16#02035;
entity("breve") -> 16#002D8;
entity("brvbar") -> 16#000A6;
entity("bscr") -> 16#1D4B7;
entity("bsemi") -> 16#0204F;
entity("bsim") -> 16#0223D;
entity("bsime") -> 16#022CD;
entity("bsol") -> 16#0005C;
entity("bsolb") -> 16#029C5;
entity("bsolhsub") -> 16#027C8;
entity("bull") -> 16#02022;
entity("bullet") -> 16#02022;
entity("bump") -> 16#0224E;
entity("bumpE") -> 16#02AAE;
entity("bumpe") -> 16#0224F;
entity("bumpeq") -> 16#0224F;
entity("cacute") -> 16#00107;
entity("cap") -> 16#02229;
entity("capand") -> 16#02A44;
entity("capbrcup") -> 16#02A49;
entity("capcap") -> 16#02A4B;
entity("capcup") -> 16#02A47;
entity("capdot") -> 16#02A40;
entity("caps") -> [16#02229, 16#0FE00];
entity("caret") -> 16#02041;
entity("caron") -> 16#002C7;
entity("ccaps") -> 16#02A4D;
entity("ccaron") -> 16#0010D;
entity("ccedil") -> 16#000E7;
entity("ccirc") -> 16#00109;
entity("ccups") -> 16#02A4C;
entity("ccupssm") -> 16#02A50;
entity("cdot") -> 16#0010B;
entity("cedil") -> 16#000B8;
entity("cemptyv") -> 16#029B2;
entity("cent") -> 16#000A2;
entity("centerdot") -> 16#000B7;
entity("cfr") -> 16#1D520;
entity("chcy") -> 16#00447;
entity("check") -> 16#02713;
entity("checkmark") -> 16#02713;
entity("chi") -> 16#003C7;
entity("cir") -> 16#025CB;
entity("cirE") -> 16#029C3;
entity("circ") -> 16#002C6;
entity("circeq") -> 16#02257;
entity("circlearrowleft") -> 16#021BA;
entity("circlearrowright") -> 16#021BB;
entity("circledR") -> 16#000AE;
entity("circledS") -> 16#024C8;
entity("circledast") -> 16#0229B;
entity("circledcirc") -> 16#0229A;
entity("circleddash") -> 16#0229D;
entity("cire") -> 16#02257;
entity("cirfnint") -> 16#02A10;
entity("cirmid") -> 16#02AEF;
entity("cirscir") -> 16#029C2;
entity("clubs") -> 16#02663;
entity("clubsuit") -> 16#02663;
entity("colon") -> 16#0003A;
entity("colone") -> 16#02254;
entity("coloneq") -> 16#02254;
entity("comma") -> 16#0002C;
entity("commat") -> 16#00040;
entity("comp") -> 16#02201;
entity("compfn") -> 16#02218;
entity("complement") -> 16#02201;
entity("complexes") -> 16#02102;
entity("cong") -> 16#02245;
entity("congdot") -> 16#02A6D;
entity("conint") -> 16#0222E;
entity("copf") -> 16#1D554;
entity("coprod") -> 16#02210;
entity("copy") -> 16#000A9;
entity("copysr") -> 16#02117;
entity("crarr") -> 16#021B5;
entity("cross") -> 16#02717;
entity("cscr") -> 16#1D4B8;
entity("csub") -> 16#02ACF;
entity("csube") -> 16#02AD1;
entity("csup") -> 16#02AD0;
entity("csupe") -> 16#02AD2;
entity("ctdot") -> 16#022EF;
entity("cudarrl") -> 16#02938;
entity("cudarrr") -> 16#02935;
entity("cuepr") -> 16#022DE;
entity("cuesc") -> 16#022DF;
entity("cularr") -> 16#021B6;
entity("cularrp") -> 16#0293D;
entity("cup") -> 16#0222A;
entity("cupbrcap") -> 16#02A48;
entity("cupcap") -> 16#02A46;
entity("cupcup") -> 16#02A4A;
entity("cupdot") -> 16#0228D;
entity("cupor") -> 16#02A45;
entity("cups") -> [16#0222A, 16#0FE00];
entity("curarr") -> 16#021B7;
entity("curarrm") -> 16#0293C;
entity("curlyeqprec") -> 16#022DE;
entity("curlyeqsucc") -> 16#022DF;
entity("curlyvee") -> 16#022CE;
entity("curlywedge") -> 16#022CF;
entity("curren") -> 16#000A4;
entity("curvearrowleft") -> 16#021B6;
entity("curvearrowright") -> 16#021B7;
entity("cuvee") -> 16#022CE;
entity("cuwed") -> 16#022CF;
entity("cwconint") -> 16#02232;
entity("cwint") -> 16#02231;
entity("cylcty") -> 16#0232D;
entity("dArr") -> 16#021D3;
entity("dHar") -> 16#02965;
entity("dagger") -> 16#02020;
entity("daleth") -> 16#02138;
entity("darr") -> 16#02193;
entity("dash") -> 16#02010;
entity("dashv") -> 16#022A3;
entity("dbkarow") -> 16#0290F;
entity("dblac") -> 16#002DD;
entity("dcaron") -> 16#0010F;
entity("dcy") -> 16#00434;
entity("dd") -> 16#02146;
entity("ddagger") -> 16#02021;
entity("ddarr") -> 16#021CA;
entity("ddotseq") -> 16#02A77;
entity("deg") -> 16#000B0;
entity("delta") -> 16#003B4;
entity("demptyv") -> 16#029B1;
entity("dfisht") -> 16#0297F;
entity("dfr") -> 16#1D521;
entity("dharl") -> 16#021C3;
entity("dharr") -> 16#021C2;
entity("diam") -> 16#022C4;
entity("diamond") -> 16#022C4;
entity("diamondsuit") -> 16#02666;
entity("diams") -> 16#02666;
entity("die") -> 16#000A8;
entity("digamma") -> 16#003DD;
entity("disin") -> 16#022F2;
entity("div") -> 16#000F7;
entity("divide") -> 16#000F7;
entity("divideontimes") -> 16#022C7;
entity("divonx") -> 16#022C7;
entity("djcy") -> 16#00452;
entity("dlcorn") -> 16#0231E;
entity("dlcrop") -> 16#0230D;
entity("dollar") -> 16#00024;
entity("dopf") -> 16#1D555;
entity("dot") -> 16#002D9;
entity("doteq") -> 16#02250;
entity("doteqdot") -> 16#02251;
entity("dotminus") -> 16#02238;
entity("dotplus") -> 16#02214;
entity("dotsquare") -> 16#022A1;
entity("doublebarwedge") -> 16#02306;
entity("downarrow") -> 16#02193;
entity("downdownarrows") -> 16#021CA;
entity("downharpoonleft") -> 16#021C3;
entity("downharpoonright") -> 16#021C2;
entity("drbkarow") -> 16#02910;
entity("drcorn") -> 16#0231F;
entity("drcrop") -> 16#0230C;
entity("dscr") -> 16#1D4B9;
entity("dscy") -> 16#00455;
entity("dsol") -> 16#029F6;
entity("dstrok") -> 16#00111;
entity("dtdot") -> 16#022F1;
entity("dtri") -> 16#025BF;
entity("dtrif") -> 16#025BE;
entity("duarr") -> 16#021F5;
entity("duhar") -> 16#0296F;
entity("dwangle") -> 16#029A6;
entity("dzcy") -> 16#0045F;
entity("dzigrarr") -> 16#027FF;
entity("eDDot") -> 16#02A77;
entity("eDot") -> 16#02251;
entity("eacute") -> 16#000E9;
entity("easter") -> 16#02A6E;
entity("ecaron") -> 16#0011B;
entity("ecir") -> 16#02256;
entity("ecirc") -> 16#000EA;
entity("ecolon") -> 16#02255;
entity("ecy") -> 16#0044D;
entity("edot") -> 16#00117;
entity("ee") -> 16#02147;
entity("efDot") -> 16#02252;
entity("efr") -> 16#1D522;
entity("eg") -> 16#02A9A;
entity("egrave") -> 16#000E8;
entity("egs") -> 16#02A96;
entity("egsdot") -> 16#02A98;
entity("el") -> 16#02A99;
entity("elinters") -> 16#023E7;
entity("ell") -> 16#02113;
entity("els") -> 16#02A95;
entity("elsdot") -> 16#02A97;
entity("emacr") -> 16#00113;
entity("empty") -> 16#02205;
entity("emptyset") -> 16#02205;
entity("emptyv") -> 16#02205;
entity("emsp") -> 16#02003;
entity("emsp13") -> 16#02004;
entity("emsp14") -> 16#02005;
entity("eng") -> 16#0014B;
entity("ensp") -> 16#02002;
entity("eogon") -> 16#00119;
entity("eopf") -> 16#1D556;
entity("epar") -> 16#022D5;
entity("eparsl") -> 16#029E3;
entity("eplus") -> 16#02A71;
entity("epsi") -> 16#003B5;
entity("epsilon") -> 16#003B5;
entity("epsiv") -> 16#003F5;
entity("eqcirc") -> 16#02256;
entity("eqcolon") -> 16#02255;
entity("eqsim") -> 16#02242;
entity("eqslantgtr") -> 16#02A96;
entity("eqslantless") -> 16#02A95;
entity("equals") -> 16#0003D;
entity("equest") -> 16#0225F;
entity("equiv") -> 16#02261;
entity("equivDD") -> 16#02A78;
entity("eqvparsl") -> 16#029E5;
entity("erDot") -> 16#02253;
entity("erarr") -> 16#02971;
entity("escr") -> 16#0212F;
entity("esdot") -> 16#02250;
entity("esim") -> 16#02242;
entity("eta") -> 16#003B7;
entity("eth") -> 16#000F0;
entity("euml") -> 16#000EB;
entity("euro") -> 16#020AC;
entity("excl") -> 16#00021;
entity("exist") -> 16#02203;
entity("expectation") -> 16#02130;
entity("exponentiale") -> 16#02147;
entity("fallingdotseq") -> 16#02252;
entity("fcy") -> 16#00444;
entity("female") -> 16#02640;
entity("ffilig") -> 16#0FB03;
entity("fflig") -> 16#0FB00;
entity("ffllig") -> 16#0FB04;
entity("ffr") -> 16#1D523;
entity("filig") -> 16#0FB01;
entity("fjlig") -> [16#00066, 16#0006A];
entity("flat") -> 16#0266D;
entity("fllig") -> 16#0FB02;
entity("fltns") -> 16#025B1;
entity("fnof") -> 16#00192;
entity("fopf") -> 16#1D557;
entity("forall") -> 16#02200;
entity("fork") -> 16#022D4;
entity("forkv") -> 16#02AD9;
entity("fpartint") -> 16#02A0D;
entity("frac12") -> 16#000BD;
entity("frac13") -> 16#02153;
entity("frac14") -> 16#000BC;
entity("frac15") -> 16#02155;
entity("frac16") -> 16#02159;
entity("frac18") -> 16#0215B;
entity("frac23") -> 16#02154;
entity("frac25") -> 16#02156;
entity("frac34") -> 16#000BE;
entity("frac35") -> 16#02157;
entity("frac38") -> 16#0215C;
entity("frac45") -> 16#02158;
entity("frac56") -> 16#0215A;
entity("frac58") -> 16#0215D;
entity("frac78") -> 16#0215E;
entity("frasl") -> 16#02044;
entity("frown") -> 16#02322;
entity("fscr") -> 16#1D4BB;
entity("gE") -> 16#02267;
entity("gEl") -> 16#02A8C;
entity("gacute") -> 16#001F5;
entity("gamma") -> 16#003B3;
entity("gammad") -> 16#003DD;
entity("gap") -> 16#02A86;
entity("gbreve") -> 16#0011F;
entity("gcirc") -> 16#0011D;
entity("gcy") -> 16#00433;
entity("gdot") -> 16#00121;
entity("ge") -> 16#02265;
entity("gel") -> 16#022DB;
entity("geq") -> 16#02265;
entity("geqq") -> 16#02267;
entity("geqslant") -> 16#02A7E;
entity("ges") -> 16#02A7E;
entity("gescc") -> 16#02AA9;
entity("gesdot") -> 16#02A80;
entity("gesdoto") -> 16#02A82;
entity("gesdotol") -> 16#02A84;
entity("gesl") -> [16#022DB, 16#0FE00];
entity("gesles") -> 16#02A94;
entity("gfr") -> 16#1D524;
entity("gg") -> 16#0226B;
entity("ggg") -> 16#022D9;
entity("gimel") -> 16#02137;
entity("gjcy") -> 16#00453;
entity("gl") -> 16#02277;
entity("glE") -> 16#02A92;
entity("gla") -> 16#02AA5;
entity("glj") -> 16#02AA4;
entity("gnE") -> 16#02269;
entity("gnap") -> 16#02A8A;
entity("gnapprox") -> 16#02A8A;
entity("gne") -> 16#02A88;
entity("gneq") -> 16#02A88;
entity("gneqq") -> 16#02269;
entity("gnsim") -> 16#022E7;
entity("gopf") -> 16#1D558;
entity("grave") -> 16#00060;
entity("gscr") -> 16#0210A;
entity("gsim") -> 16#02273;
entity("gsime") -> 16#02A8E;
entity("gsiml") -> 16#02A90;
entity("gt") -> 16#0003E;
entity("gtcc") -> 16#02AA7;
entity("gtcir") -> 16#02A7A;
entity("gtdot") -> 16#022D7;
entity("gtlPar") -> 16#02995;
entity("gtquest") -> 16#02A7C;
entity("gtrapprox") -> 16#02A86;
entity("gtrarr") -> 16#02978;
entity("gtrdot") -> 16#022D7;
entity("gtreqless") -> 16#022DB;
entity("gtreqqless") -> 16#02A8C;
entity("gtrless") -> 16#02277;
entity("gtrsim") -> 16#02273;
entity("gvertneqq") -> [16#02269, 16#0FE00];
entity("gvnE") -> [16#02269, 16#0FE00];
entity("hArr") -> 16#021D4;
entity("hairsp") -> 16#0200A;
entity("half") -> 16#000BD;
entity("hamilt") -> 16#0210B;
entity("hardcy") -> 16#0044A;
entity("harr") -> 16#02194;
entity("harrcir") -> 16#02948;
entity("harrw") -> 16#021AD;
entity("hbar") -> 16#0210F;
entity("hcirc") -> 16#00125;
entity("hearts") -> 16#02665;
entity("heartsuit") -> 16#02665;
entity("hellip") -> 16#02026;
entity("hercon") -> 16#022B9;
entity("hfr") -> 16#1D525;
entity("hksearow") -> 16#02925;
entity("hkswarow") -> 16#02926;
entity("hoarr") -> 16#021FF;
entity("homtht") -> 16#0223B;
entity("hookleftarrow") -> 16#021A9;
entity("hookrightarrow") -> 16#021AA;
entity("hopf") -> 16#1D559;
entity("horbar") -> 16#02015;
entity("hscr") -> 16#1D4BD;
entity("hslash") -> 16#0210F;
entity("hstrok") -> 16#00127;
entity("hybull") -> 16#02043;
entity("hyphen") -> 16#02010;
entity("iacute") -> 16#000ED;
entity("ic") -> 16#02063;
entity("icirc") -> 16#000EE;
entity("icy") -> 16#00438;
entity("iecy") -> 16#00435;
entity("iexcl") -> 16#000A1;
entity("iff") -> 16#021D4;
entity("ifr") -> 16#1D526;
entity("igrave") -> 16#000EC;
entity("ii") -> 16#02148;
entity("iiiint") -> 16#02A0C;
entity("iiint") -> 16#0222D;
entity("iinfin") -> 16#029DC;
entity("iiota") -> 16#02129;
entity("ijlig") -> 16#00133;
entity("imacr") -> 16#0012B;
entity("image") -> 16#02111;
entity("imagline") -> 16#02110;
entity("imagpart") -> 16#02111;
entity("imath") -> 16#00131;
entity("imof") -> 16#022B7;
entity("imped") -> 16#001B5;
entity("in") -> 16#02208;
entity("incare") -> 16#02105;
entity("infin") -> 16#0221E;
entity("infintie") -> 16#029DD;
entity("inodot") -> 16#00131;
entity("int") -> 16#0222B;
entity("intcal") -> 16#022BA;
entity("integers") -> 16#02124;
entity("intercal") -> 16#022BA;
entity("intlarhk") -> 16#02A17;
entity("intprod") -> 16#02A3C;
entity("iocy") -> 16#00451;
entity("iogon") -> 16#0012F;
entity("iopf") -> 16#1D55A;
entity("iota") -> 16#003B9;
entity("iprod") -> 16#02A3C;
entity("iquest") -> 16#000BF;
entity("iscr") -> 16#1D4BE;
entity("isin") -> 16#02208;
entity("isinE") -> 16#022F9;
entity("isindot") -> 16#022F5;
entity("isins") -> 16#022F4;
entity("isinsv") -> 16#022F3;
entity("isinv") -> 16#02208;
entity("it") -> 16#02062;
entity("itilde") -> 16#00129;
entity("iukcy") -> 16#00456;
entity("iuml") -> 16#000EF;
entity("jcirc") -> 16#00135;
entity("jcy") -> 16#00439;
entity("jfr") -> 16#1D527;
entity("jmath") -> 16#00237;
entity("jopf") -> 16#1D55B;
entity("jscr") -> 16#1D4BF;
entity("jsercy") -> 16#00458;
entity("jukcy") -> 16#00454;
entity("kappa") -> 16#003BA;
entity("kappav") -> 16#003F0;
entity("kcedil") -> 16#00137;
entity("kcy") -> 16#0043A;
entity("kfr") -> 16#1D528;
entity("kgreen") -> 16#00138;
entity("khcy") -> 16#00445;
entity("kjcy") -> 16#0045C;
entity("kopf") -> 16#1D55C;
entity("kscr") -> 16#1D4C0;
entity("lAarr") -> 16#021DA;
entity("lArr") -> 16#021D0;
entity("lAtail") -> 16#0291B;
entity("lBarr") -> 16#0290E;
entity("lE") -> 16#02266;
entity("lEg") -> 16#02A8B;
entity("lHar") -> 16#02962;
entity("lacute") -> 16#0013A;
entity("laemptyv") -> 16#029B4;
entity("lagran") -> 16#02112;
entity("lambda") -> 16#003BB;
entity("lang") -> 16#027E8;
entity("langd") -> 16#02991;
entity("langle") -> 16#027E8;
entity("lap") -> 16#02A85;
entity("laquo") -> 16#000AB;
entity("larr") -> 16#02190;
entity("larrb") -> 16#021E4;
entity("larrbfs") -> 16#0291F;
entity("larrfs") -> 16#0291D;
entity("larrhk") -> 16#021A9;
entity("larrlp") -> 16#021AB;
entity("larrpl") -> 16#02939;
entity("larrsim") -> 16#02973;
entity("larrtl") -> 16#021A2;
entity("lat") -> 16#02AAB;
entity("latail") -> 16#02919;
entity("late") -> 16#02AAD;
entity("lates") -> [16#02AAD, 16#0FE00];
entity("lbarr") -> 16#0290C;
entity("lbbrk") -> 16#02772;
entity("lbrace") -> 16#0007B;
entity("lbrack") -> 16#0005B;
entity("lbrke") -> 16#0298B;
entity("lbrksld") -> 16#0298F;
entity("lbrkslu") -> 16#0298D;
entity("lcaron") -> 16#0013E;
entity("lcedil") -> 16#0013C;
entity("lceil") -> 16#02308;
entity("lcub") -> 16#0007B;
entity("lcy") -> 16#0043B;
entity("ldca") -> 16#02936;
entity("ldquo") -> 16#0201C;
entity("ldquor") -> 16#0201E;
entity("ldrdhar") -> 16#02967;
entity("ldrushar") -> 16#0294B;
entity("ldsh") -> 16#021B2;
entity("le") -> 16#02264;
entity("leftarrow") -> 16#02190;
entity("leftarrowtail") -> 16#021A2;
entity("leftharpoondown") -> 16#021BD;
entity("leftharpoonup") -> 16#021BC;
entity("leftleftarrows") -> 16#021C7;
entity("leftrightarrow") -> 16#02194;
entity("leftrightarrows") -> 16#021C6;
entity("leftrightharpoons") -> 16#021CB;
entity("leftrightsquigarrow") -> 16#021AD;
entity("leftthreetimes") -> 16#022CB;
entity("leg") -> 16#022DA;
entity("leq") -> 16#02264;
entity("leqq") -> 16#02266;
entity("leqslant") -> 16#02A7D;
entity("les") -> 16#02A7D;
entity("lescc") -> 16#02AA8;
entity("lesdot") -> 16#02A7F;
entity("lesdoto") -> 16#02A81;
entity("lesdotor") -> 16#02A83;
entity("lesg") -> [16#022DA, 16#0FE00];
entity("lesges") -> 16#02A93;
entity("lessapprox") -> 16#02A85;
entity("lessdot") -> 16#022D6;
entity("lesseqgtr") -> 16#022DA;
entity("lesseqqgtr") -> 16#02A8B;
entity("lessgtr") -> 16#02276;
entity("lesssim") -> 16#02272;
entity("lfisht") -> 16#0297C;
entity("lfloor") -> 16#0230A;
entity("lfr") -> 16#1D529;
entity("lg") -> 16#02276;
entity("lgE") -> 16#02A91;
entity("lhard") -> 16#021BD;
entity("lharu") -> 16#021BC;
entity("lharul") -> 16#0296A;
entity("lhblk") -> 16#02584;
entity("ljcy") -> 16#00459;
entity("ll") -> 16#0226A;
entity("llarr") -> 16#021C7;
entity("llcorner") -> 16#0231E;
entity("llhard") -> 16#0296B;
entity("lltri") -> 16#025FA;
entity("lmidot") -> 16#00140;
entity("lmoust") -> 16#023B0;
entity("lmoustache") -> 16#023B0;
entity("lnE") -> 16#02268;
entity("lnap") -> 16#02A89;
entity("lnapprox") -> 16#02A89;
entity("lne") -> 16#02A87;
entity("lneq") -> 16#02A87;
entity("lneqq") -> 16#02268;
entity("lnsim") -> 16#022E6;
entity("loang") -> 16#027EC;
entity("loarr") -> 16#021FD;
entity("lobrk") -> 16#027E6;
entity("longleftarrow") -> 16#027F5;
entity("longleftrightarrow") -> 16#027F7;
entity("longmapsto") -> 16#027FC;
entity("longrightarrow") -> 16#027F6;
entity("looparrowleft") -> 16#021AB;
entity("looparrowright") -> 16#021AC;
entity("lopar") -> 16#02985;
entity("lopf") -> 16#1D55D;
entity("loplus") -> 16#02A2D;
entity("lotimes") -> 16#02A34;
entity("lowast") -> 16#02217;
entity("lowbar") -> 16#0005F;
entity("loz") -> 16#025CA;
entity("lozenge") -> 16#025CA;
entity("lozf") -> 16#029EB;
entity("lpar") -> 16#00028;
entity("lparlt") -> 16#02993;
entity("lrarr") -> 16#021C6;
entity("lrcorner") -> 16#0231F;
entity("lrhar") -> 16#021CB;
entity("lrhard") -> 16#0296D;
entity("lrm") -> 16#0200E;
entity("lrtri") -> 16#022BF;
entity("lsaquo") -> 16#02039;
entity("lscr") -> 16#1D4C1;
entity("lsh") -> 16#021B0;
entity("lsim") -> 16#02272;
entity("lsime") -> 16#02A8D;
entity("lsimg") -> 16#02A8F;
entity("lsqb") -> 16#0005B;
entity("lsquo") -> 16#02018;
entity("lsquor") -> 16#0201A;
entity("lstrok") -> 16#00142;
entity("lt") -> 16#0003C;
entity("ltcc") -> 16#02AA6;
entity("ltcir") -> 16#02A79;
entity("ltdot") -> 16#022D6;
entity("lthree") -> 16#022CB;
entity("ltimes") -> 16#022C9;
entity("ltlarr") -> 16#02976;
entity("ltquest") -> 16#02A7B;
entity("ltrPar") -> 16#02996;
entity("ltri") -> 16#025C3;
entity("ltrie") -> 16#022B4;
entity("ltrif") -> 16#025C2;
entity("lurdshar") -> 16#0294A;
entity("luruhar") -> 16#02966;
entity("lvertneqq") -> [16#02268, 16#0FE00];
entity("lvnE") -> [16#02268, 16#0FE00];
entity("mDDot") -> 16#0223A;
entity("macr") -> 16#000AF;
entity("male") -> 16#02642;
entity("malt") -> 16#02720;
entity("maltese") -> 16#02720;
entity("map") -> 16#021A6;
entity("mapsto") -> 16#021A6;
entity("mapstodown") -> 16#021A7;
entity("mapstoleft") -> 16#021A4;
entity("mapstoup") -> 16#021A5;
entity("marker") -> 16#025AE;
entity("mcomma") -> 16#02A29;
entity("mcy") -> 16#0043C;
entity("mdash") -> 16#02014;
entity("measuredangle") -> 16#02221;
entity("mfr") -> 16#1D52A;
entity("mho") -> 16#02127;
entity("micro") -> 16#000B5;
entity("mid") -> 16#02223;
entity("midast") -> 16#0002A;
entity("midcir") -> 16#02AF0;
entity("middot") -> 16#000B7;
entity("minus") -> 16#02212;
entity("minusb") -> 16#0229F;
entity("minusd") -> 16#02238;
entity("minusdu") -> 16#02A2A;
entity("mlcp") -> 16#02ADB;
entity("mldr") -> 16#02026;
entity("mnplus") -> 16#02213;
entity("models") -> 16#022A7;
entity("mopf") -> 16#1D55E;
entity("mp") -> 16#02213;
entity("mscr") -> 16#1D4C2;
entity("mstpos") -> 16#0223E;
entity("mu") -> 16#003BC;
entity("multimap") -> 16#022B8;
entity("mumap") -> 16#022B8;
entity("nGg") -> [16#022D9, 16#00338];
entity("nGt") -> [16#0226B, 16#020D2];
entity("nGtv") -> [16#0226B, 16#00338];
entity("nLeftarrow") -> 16#021CD;
entity("nLeftrightarrow") -> 16#021CE;
entity("nLl") -> [16#022D8, 16#00338];
entity("nLt") -> [16#0226A, 16#020D2];
entity("nLtv") -> [16#0226A, 16#00338];
entity("nRightarrow") -> 16#021CF;
entity("nVDash") -> 16#022AF;
entity("nVdash") -> 16#022AE;
entity("nabla") -> 16#02207;
entity("nacute") -> 16#00144;
entity("nang") -> [16#02220, 16#020D2];
entity("nap") -> 16#02249;
entity("napE") -> [16#02A70, 16#00338];
entity("napid") -> [16#0224B, 16#00338];
entity("napos") -> 16#00149;
entity("napprox") -> 16#02249;
entity("natur") -> 16#0266E;
entity("natural") -> 16#0266E;
entity("naturals") -> 16#02115;
entity("nbsp") -> 16#000A0;
entity("nbump") -> [16#0224E, 16#00338];
entity("nbumpe") -> [16#0224F, 16#00338];
entity("ncap") -> 16#02A43;
entity("ncaron") -> 16#00148;
entity("ncedil") -> 16#00146;
entity("ncong") -> 16#02247;
entity("ncongdot") -> [16#02A6D, 16#00338];
entity("ncup") -> 16#02A42;
entity("ncy") -> 16#0043D;
entity("ndash") -> 16#02013;
entity("ne") -> 16#02260;
entity("neArr") -> 16#021D7;
entity("nearhk") -> 16#02924;
entity("nearr") -> 16#02197;
entity("nearrow") -> 16#02197;
entity("nedot") -> [16#02250, 16#00338];
entity("nequiv") -> 16#02262;
entity("nesear") -> 16#02928;
entity("nesim") -> [16#02242, 16#00338];
entity("nexist") -> 16#02204;
entity("nexists") -> 16#02204;
entity("nfr") -> 16#1D52B;
entity("ngE") -> [16#02267, 16#00338];
entity("nge") -> 16#02271;
entity("ngeq") -> 16#02271;
entity("ngeqq") -> [16#02267, 16#00338];
entity("ngeqslant") -> [16#02A7E, 16#00338];
entity("nges") -> [16#02A7E, 16#00338];
entity("ngsim") -> 16#02275;
entity("ngt") -> 16#0226F;
entity("ngtr") -> 16#0226F;
entity("nhArr") -> 16#021CE;
entity("nharr") -> 16#021AE;
entity("nhpar") -> 16#02AF2;
entity("ni") -> 16#0220B;
entity("nis") -> 16#022FC;
entity("nisd") -> 16#022FA;
entity("niv") -> 16#0220B;
entity("njcy") -> 16#0045A;
entity("nlArr") -> 16#021CD;
entity("nlE") -> [16#02266, 16#00338];
entity("nlarr") -> 16#0219A;
entity("nldr") -> 16#02025;
entity("nle") -> 16#02270;
entity("nleftarrow") -> 16#0219A;
entity("nleftrightarrow") -> 16#021AE;
entity("nleq") -> 16#02270;
entity("nleqq") -> [16#02266, 16#00338];
entity("nleqslant") -> [16#02A7D, 16#00338];
entity("nles") -> [16#02A7D, 16#00338];
entity("nless") -> 16#0226E;
entity("nlsim") -> 16#02274;
entity("nlt") -> 16#0226E;
entity("nltri") -> 16#022EA;
entity("nltrie") -> 16#022EC;
entity("nmid") -> 16#02224;
entity("nopf") -> 16#1D55F;
entity("not") -> 16#000AC;
entity("notin") -> 16#02209;
entity("notinE") -> [16#022F9, 16#00338];
entity("notindot") -> [16#022F5, 16#00338];
entity("notinva") -> 16#02209;
entity("notinvb") -> 16#022F7;
entity("notinvc") -> 16#022F6;
entity("notni") -> 16#0220C;
entity("notniva") -> 16#0220C;
entity("notnivb") -> 16#022FE;
entity("notnivc") -> 16#022FD;
entity("npar") -> 16#02226;
entity("nparallel") -> 16#02226;
entity("nparsl") -> [16#02AFD, 16#020E5];
entity("npart") -> [16#02202, 16#00338];
entity("npolint") -> 16#02A14;
entity("npr") -> 16#02280;
entity("nprcue") -> 16#022E0;
entity("npre") -> [16#02AAF, 16#00338];
entity("nprec") -> 16#02280;
entity("npreceq") -> [16#02AAF, 16#00338];
entity("nrArr") -> 16#021CF;
entity("nrarr") -> 16#0219B;
entity("nrarrc") -> [16#02933, 16#00338];
entity("nrarrw") -> [16#0219D, 16#00338];
entity("nrightarrow") -> 16#0219B;
entity("nrtri") -> 16#022EB;
entity("nrtrie") -> 16#022ED;
entity("nsc") -> 16#02281;
entity("nsccue") -> 16#022E1;
entity("nsce") -> [16#02AB0, 16#00338];
entity("nscr") -> 16#1D4C3;
entity("nshortmid") -> 16#02224;
entity("nshortparallel") -> 16#02226;
entity("nsim") -> 16#02241;
entity("nsime") -> 16#02244;
entity("nsimeq") -> 16#02244;
entity("nsmid") -> 16#02224;
entity("nspar") -> 16#02226;
entity("nsqsube") -> 16#022E2;
entity("nsqsupe") -> 16#022E3;
entity("nsub") -> 16#02284;
entity("nsubE") -> [16#02AC5, 16#00338];
entity("nsube") -> 16#02288;
entity("nsubset") -> [16#02282, 16#020D2];
entity("nsubseteq") -> 16#02288;
entity("nsubseteqq") -> [16#02AC5, 16#00338];
entity("nsucc") -> 16#02281;
entity("nsucceq") -> [16#02AB0, 16#00338];
entity("nsup") -> 16#02285;
entity("nsupE") -> [16#02AC6, 16#00338];
entity("nsupe") -> 16#02289;
entity("nsupset") -> [16#02283, 16#020D2];
entity("nsupseteq") -> 16#02289;
entity("nsupseteqq") -> [16#02AC6, 16#00338];
entity("ntgl") -> 16#02279;
entity("ntilde") -> 16#000F1;
entity("ntlg") -> 16#02278;
entity("ntriangleleft") -> 16#022EA;
entity("ntrianglelefteq") -> 16#022EC;
entity("ntriangleright") -> 16#022EB;
entity("ntrianglerighteq") -> 16#022ED;
entity("nu") -> 16#003BD;
entity("num") -> 16#00023;
entity("numero") -> 16#02116;
entity("numsp") -> 16#02007;
entity("nvDash") -> 16#022AD;
entity("nvHarr") -> 16#02904;
entity("nvap") -> [16#0224D, 16#020D2];
entity("nvdash") -> 16#022AC;
entity("nvge") -> [16#02265, 16#020D2];
entity("nvgt") -> [16#0003E, 16#020D2];
entity("nvinfin") -> 16#029DE;
entity("nvlArr") -> 16#02902;
entity("nvle") -> [16#02264, 16#020D2];
entity("nvlt") -> [16#0003C, 16#020D2];
entity("nvltrie") -> [16#022B4, 16#020D2];
entity("nvrArr") -> 16#02903;
entity("nvrtrie") -> [16#022B5, 16#020D2];
entity("nvsim") -> [16#0223C, 16#020D2];
entity("nwArr") -> 16#021D6;
entity("nwarhk") -> 16#02923;
entity("nwarr") -> 16#02196;
entity("nwarrow") -> 16#02196;
entity("nwnear") -> 16#02927;
entity("oS") -> 16#024C8;
entity("oacute") -> 16#000F3;
entity("oast") -> 16#0229B;
entity("ocir") -> 16#0229A;
entity("ocirc") -> 16#000F4;
entity("ocy") -> 16#0043E;
entity("odash") -> 16#0229D;
entity("odblac") -> 16#00151;
entity("odiv") -> 16#02A38;
entity("odot") -> 16#02299;
entity("odsold") -> 16#029BC;
entity("oelig") -> 16#00153;
entity("ofcir") -> 16#029BF;
entity("ofr") -> 16#1D52C;
entity("ogon") -> 16#002DB;
entity("ograve") -> 16#000F2;
entity("ogt") -> 16#029C1;
entity("ohbar") -> 16#029B5;
entity("ohm") -> 16#003A9;
entity("oint") -> 16#0222E;
entity("olarr") -> 16#021BA;
entity("olcir") -> 16#029BE;
entity("olcross") -> 16#029BB;
entity("oline") -> 16#0203E;
entity("olt") -> 16#029C0;
entity("omacr") -> 16#0014D;
entity("omega") -> 16#003C9;
entity("omicron") -> 16#003BF;
entity("omid") -> 16#029B6;
entity("ominus") -> 16#02296;
entity("oopf") -> 16#1D560;
entity("opar") -> 16#029B7;
entity("operp") -> 16#029B9;
entity("oplus") -> 16#02295;
entity("or") -> 16#02228;
entity("orarr") -> 16#021BB;
entity("ord") -> 16#02A5D;
entity("order") -> 16#02134;
entity("orderof") -> 16#02134;
entity("ordf") -> 16#000AA;
entity("ordm") -> 16#000BA;
entity("origof") -> 16#022B6;
entity("oror") -> 16#02A56;
entity("orslope") -> 16#02A57;
entity("orv") -> 16#02A5B;
entity("oscr") -> 16#02134;
entity("oslash") -> 16#000F8;
entity("osol") -> 16#02298;
entity("otilde") -> 16#000F5;
entity("otimes") -> 16#02297;
entity("otimesas") -> 16#02A36;
entity("ouml") -> 16#000F6;
entity("ovbar") -> 16#0233D;
entity("par") -> 16#02225;
entity("para") -> 16#000B6;
entity("parallel") -> 16#02225;
entity("parsim") -> 16#02AF3;
entity("parsl") -> 16#02AFD;
entity("part") -> 16#02202;
entity("pcy") -> 16#0043F;
entity("percnt") -> 16#00025;
entity("period") -> 16#0002E;
entity("permil") -> 16#02030;
entity("perp") -> 16#022A5;
entity("pertenk") -> 16#02031;
entity("pfr") -> 16#1D52D;
entity("phi") -> 16#003C6;
entity("phiv") -> 16#003D5;
entity("phmmat") -> 16#02133;
entity("phone") -> 16#0260E;
entity("pi") -> 16#003C0;
entity("pitchfork") -> 16#022D4;
entity("piv") -> 16#003D6;
entity("planck") -> 16#0210F;
entity("planckh") -> 16#0210E;
entity("plankv") -> 16#0210F;
entity("plus") -> 16#0002B;
entity("plusacir") -> 16#02A23;
entity("plusb") -> 16#0229E;
entity("pluscir") -> 16#02A22;
entity("plusdo") -> 16#02214;
entity("plusdu") -> 16#02A25;
entity("pluse") -> 16#02A72;
entity("plusmn") -> 16#000B1;
entity("plussim") -> 16#02A26;
entity("plustwo") -> 16#02A27;
entity("pm") -> 16#000B1;
entity("pointint") -> 16#02A15;
entity("popf") -> 16#1D561;
entity("pound") -> 16#000A3;
entity("pr") -> 16#0227A;
entity("prE") -> 16#02AB3;
entity("prap") -> 16#02AB7;
entity("prcue") -> 16#0227C;
entity("pre") -> 16#02AAF;
entity("prec") -> 16#0227A;
entity("precapprox") -> 16#02AB7;
entity("preccurlyeq") -> 16#0227C;
entity("preceq") -> 16#02AAF;
entity("precnapprox") -> 16#02AB9;
entity("precneqq") -> 16#02AB5;
entity("precnsim") -> 16#022E8;
entity("precsim") -> 16#0227E;
entity("prime") -> 16#02032;
entity("primes") -> 16#02119;
entity("prnE") -> 16#02AB5;
entity("prnap") -> 16#02AB9;
entity("prnsim") -> 16#022E8;
entity("prod") -> 16#0220F;
entity("profalar") -> 16#0232E;
entity("profline") -> 16#02312;
entity("profsurf") -> 16#02313;
entity("prop") -> 16#0221D;
entity("propto") -> 16#0221D;
entity("prsim") -> 16#0227E;
entity("prurel") -> 16#022B0;
entity("pscr") -> 16#1D4C5;
entity("psi") -> 16#003C8;
entity("puncsp") -> 16#02008;
entity("qfr") -> 16#1D52E;
entity("qint") -> 16#02A0C;
entity("qopf") -> 16#1D562;
entity("qprime") -> 16#02057;
entity("qscr") -> 16#1D4C6;
entity("quaternions") -> 16#0210D;
entity("quatint") -> 16#02A16;
entity("quest") -> 16#0003F;
entity("questeq") -> 16#0225F;
entity("quot") -> 16#00022;
entity("rAarr") -> 16#021DB;
entity("rArr") -> 16#021D2;
entity("rAtail") -> 16#0291C;
entity("rBarr") -> 16#0290F;
entity("rHar") -> 16#02964;
entity("race") -> [16#0223D, 16#00331];
entity("racute") -> 16#00155;
entity("radic") -> 16#0221A;
entity("raemptyv") -> 16#029B3;
entity("rang") -> 16#027E9;
entity("rangd") -> 16#02992;
entity("range") -> 16#029A5;
entity("rangle") -> 16#027E9;
entity("raquo") -> 16#000BB;
entity("rarr") -> 16#02192;
entity("rarrap") -> 16#02975;
entity("rarrb") -> 16#021E5;
entity("rarrbfs") -> 16#02920;
entity("rarrc") -> 16#02933;
entity("rarrfs") -> 16#0291E;
entity("rarrhk") -> 16#021AA;
entity("rarrlp") -> 16#021AC;
entity("rarrpl") -> 16#02945;
entity("rarrsim") -> 16#02974;
entity("rarrtl") -> 16#021A3;
entity("rarrw") -> 16#0219D;
entity("ratail") -> 16#0291A;
entity("ratio") -> 16#02236;
entity("rationals") -> 16#0211A;
entity("rbarr") -> 16#0290D;
entity("rbbrk") -> 16#02773;
entity("rbrace") -> 16#0007D;
entity("rbrack") -> 16#0005D;
entity("rbrke") -> 16#0298C;
entity("rbrksld") -> 16#0298E;
entity("rbrkslu") -> 16#02990;
entity("rcaron") -> 16#00159;
entity("rcedil") -> 16#00157;
entity("rceil") -> 16#02309;
entity("rcub") -> 16#0007D;
entity("rcy") -> 16#00440;
entity("rdca") -> 16#02937;
entity("rdldhar") -> 16#02969;
entity("rdquo") -> 16#0201D;
entity("rdquor") -> 16#0201D;
entity("rdsh") -> 16#021B3;
entity("real") -> 16#0211C;
entity("realine") -> 16#0211B;
entity("realpart") -> 16#0211C;
entity("reals") -> 16#0211D;
entity("rect") -> 16#025AD;
entity("reg") -> 16#000AE;
entity("rfisht") -> 16#0297D;
entity("rfloor") -> 16#0230B;
entity("rfr") -> 16#1D52F;
entity("rhard") -> 16#021C1;
entity("rharu") -> 16#021C0;
entity("rharul") -> 16#0296C;
entity("rho") -> 16#003C1;
entity("rhov") -> 16#003F1;
entity("rightarrow") -> 16#02192;
entity("rightarrowtail") -> 16#021A3;
entity("rightharpoondown") -> 16#021C1;
entity("rightharpoonup") -> 16#021C0;
entity("rightleftarrows") -> 16#021C4;
entity("rightleftharpoons") -> 16#021CC;
entity("rightrightarrows") -> 16#021C9;
entity("rightsquigarrow") -> 16#0219D;
entity("rightthreetimes") -> 16#022CC;
entity("ring") -> 16#002DA;
entity("risingdotseq") -> 16#02253;
entity("rlarr") -> 16#021C4;
entity("rlhar") -> 16#021CC;
entity("rlm") -> 16#0200F;
entity("rmoust") -> 16#023B1;
entity("rmoustache") -> 16#023B1;
entity("rnmid") -> 16#02AEE;
entity("roang") -> 16#027ED;
entity("roarr") -> 16#021FE;
entity("robrk") -> 16#027E7;
entity("ropar") -> 16#02986;
entity("ropf") -> 16#1D563;
entity("roplus") -> 16#02A2E;
entity("rotimes") -> 16#02A35;
entity("rpar") -> 16#00029;
entity("rpargt") -> 16#02994;
entity("rppolint") -> 16#02A12;
entity("rrarr") -> 16#021C9;
entity("rsaquo") -> 16#0203A;
entity("rscr") -> 16#1D4C7;
entity("rsh") -> 16#021B1;
entity("rsqb") -> 16#0005D;
entity("rsquo") -> 16#02019;
entity("rsquor") -> 16#02019;
entity("rthree") -> 16#022CC;
entity("rtimes") -> 16#022CA;
entity("rtri") -> 16#025B9;
entity("rtrie") -> 16#022B5;
entity("rtrif") -> 16#025B8;
entity("rtriltri") -> 16#029CE;
entity("ruluhar") -> 16#02968;
entity("rx") -> 16#0211E;
entity("sacute") -> 16#0015B;
entity("sbquo") -> 16#0201A;
entity("sc") -> 16#0227B;
entity("scE") -> 16#02AB4;
entity("scap") -> 16#02AB8;
entity("scaron") -> 16#00161;
entity("sccue") -> 16#0227D;
entity("sce") -> 16#02AB0;
entity("scedil") -> 16#0015F;
entity("scirc") -> 16#0015D;
entity("scnE") -> 16#02AB6;
entity("scnap") -> 16#02ABA;
entity("scnsim") -> 16#022E9;
entity("scpolint") -> 16#02A13;
entity("scsim") -> 16#0227F;
entity("scy") -> 16#00441;
entity("sdot") -> 16#022C5;
entity("sdotb") -> 16#022A1;
entity("sdote") -> 16#02A66;
entity("seArr") -> 16#021D8;
entity("searhk") -> 16#02925;
entity("searr") -> 16#02198;
entity("searrow") -> 16#02198;
entity("sect") -> 16#000A7;
entity("semi") -> 16#0003B;
entity("seswar") -> 16#02929;
entity("setminus") -> 16#02216;
entity("setmn") -> 16#02216;
entity("sext") -> 16#02736;
entity("sfr") -> 16#1D530;
entity("sfrown") -> 16#02322;
entity("sharp") -> 16#0266F;
entity("shchcy") -> 16#00449;
entity("shcy") -> 16#00448;
entity("shortmid") -> 16#02223;
entity("shortparallel") -> 16#02225;
entity("shy") -> 16#000AD;
entity("sigma") -> 16#003C3;
entity("sigmaf") -> 16#003C2;
entity("sigmav") -> 16#003C2;
entity("sim") -> 16#0223C;
entity("simdot") -> 16#02A6A;
entity("sime") -> 16#02243;
entity("simeq") -> 16#02243;
entity("simg") -> 16#02A9E;
entity("simgE") -> 16#02AA0;
entity("siml") -> 16#02A9D;
entity("simlE") -> 16#02A9F;
entity("simne") -> 16#02246;
entity("simplus") -> 16#02A24;
entity("simrarr") -> 16#02972;
entity("slarr") -> 16#02190;
entity("smallsetminus") -> 16#02216;
entity("smashp") -> 16#02A33;
entity("smeparsl") -> 16#029E4;
entity("smid") -> 16#02223;
entity("smile") -> 16#02323;
entity("smt") -> 16#02AAA;
entity("smte") -> 16#02AAC;
entity("smtes") -> [16#02AAC, 16#0FE00];
entity("softcy") -> 16#0044C;
entity("sol") -> 16#0002F;
entity("solb") -> 16#029C4;
entity("solbar") -> 16#0233F;
entity("sopf") -> 16#1D564;
entity("spades") -> 16#02660;
entity("spadesuit") -> 16#02660;
entity("spar") -> 16#02225;
entity("sqcap") -> 16#02293;
entity("sqcaps") -> [16#02293, 16#0FE00];
entity("sqcup") -> 16#02294;
entity("sqcups") -> [16#02294, 16#0FE00];
entity("sqsub") -> 16#0228F;
entity("sqsube") -> 16#02291;
entity("sqsubset") -> 16#0228F;
entity("sqsubseteq") -> 16#02291;
entity("sqsup") -> 16#02290;
entity("sqsupe") -> 16#02292;
entity("sqsupset") -> 16#02290;
entity("sqsupseteq") -> 16#02292;
entity("squ") -> 16#025A1;
entity("square") -> 16#025A1;
entity("squarf") -> 16#025AA;
entity("squf") -> 16#025AA;
entity("srarr") -> 16#02192;
entity("sscr") -> 16#1D4C8;
entity("ssetmn") -> 16#02216;
entity("ssmile") -> 16#02323;
entity("sstarf") -> 16#022C6;
entity("star") -> 16#02606;
entity("starf") -> 16#02605;
entity("straightepsilon") -> 16#003F5;
entity("straightphi") -> 16#003D5;
entity("strns") -> 16#000AF;
entity("sub") -> 16#02282;
entity("subE") -> 16#02AC5;
entity("subdot") -> 16#02ABD;
entity("sube") -> 16#02286;
entity("subedot") -> 16#02AC3;
entity("submult") -> 16#02AC1;
entity("subnE") -> 16#02ACB;
entity("subne") -> 16#0228A;
entity("subplus") -> 16#02ABF;
entity("subrarr") -> 16#02979;
entity("subset") -> 16#02282;
entity("subseteq") -> 16#02286;
entity("subseteqq") -> 16#02AC5;
entity("subsetneq") -> 16#0228A;
entity("subsetneqq") -> 16#02ACB;
entity("subsim") -> 16#02AC7;
entity("subsub") -> 16#02AD5;
entity("subsup") -> 16#02AD3;
entity("succ") -> 16#0227B;
entity("succapprox") -> 16#02AB8;
entity("succcurlyeq") -> 16#0227D;
entity("succeq") -> 16#02AB0;
entity("succnapprox") -> 16#02ABA;
entity("succneqq") -> 16#02AB6;
entity("succnsim") -> 16#022E9;
entity("succsim") -> 16#0227F;
entity("sum") -> 16#02211;
entity("sung") -> 16#0266A;
entity("sup") -> 16#02283;
entity("sup1") -> 16#000B9;
entity("sup2") -> 16#000B2;
entity("sup3") -> 16#000B3;
entity("supE") -> 16#02AC6;
entity("supdot") -> 16#02ABE;
entity("supdsub") -> 16#02AD8;
entity("supe") -> 16#02287;
entity("supedot") -> 16#02AC4;
entity("suphsol") -> 16#027C9;
entity("suphsub") -> 16#02AD7;
entity("suplarr") -> 16#0297B;
entity("supmult") -> 16#02AC2;
entity("supnE") -> 16#02ACC;
entity("supne") -> 16#0228B;
entity("supplus") -> 16#02AC0;
entity("supset") -> 16#02283;
entity("supseteq") -> 16#02287;
entity("supseteqq") -> 16#02AC6;
entity("supsetneq") -> 16#0228B;
entity("supsetneqq") -> 16#02ACC;
entity("supsim") -> 16#02AC8;
entity("supsub") -> 16#02AD4;
entity("supsup") -> 16#02AD6;
entity("swArr") -> 16#021D9;
entity("swarhk") -> 16#02926;
entity("swarr") -> 16#02199;
entity("swarrow") -> 16#02199;
entity("swnwar") -> 16#0292A;
entity("szlig") -> 16#000DF;
entity("target") -> 16#02316;
entity("tau") -> 16#003C4;
entity("tbrk") -> 16#023B4;
entity("tcaron") -> 16#00165;
entity("tcedil") -> 16#00163;
entity("tcy") -> 16#00442;
entity("tdot") -> 16#020DB;
entity("telrec") -> 16#02315;
entity("tfr") -> 16#1D531;
entity("there4") -> 16#02234;
entity("therefore") -> 16#02234;
entity("theta") -> 16#003B8;
entity("thetasym") -> 16#003D1;
entity("thetav") -> 16#003D1;
entity("thickapprox") -> 16#02248;
entity("thicksim") -> 16#0223C;
entity("thinsp") -> 16#02009;
entity("thkap") -> 16#02248;
entity("thksim") -> 16#0223C;
entity("thorn") -> 16#000FE;
entity("tilde") -> 16#002DC;
entity("times") -> 16#000D7;
entity("timesb") -> 16#022A0;
entity("timesbar") -> 16#02A31;
entity("timesd") -> 16#02A30;
entity("tint") -> 16#0222D;
entity("toea") -> 16#02928;
entity("top") -> 16#022A4;
entity("topbot") -> 16#02336;
entity("topcir") -> 16#02AF1;
entity("topf") -> 16#1D565;
entity("topfork") -> 16#02ADA;
entity("tosa") -> 16#02929;
entity("tprime") -> 16#02034;
entity("trade") -> 16#02122;
entity("triangle") -> 16#025B5;
entity("triangledown") -> 16#025BF;
entity("triangleleft") -> 16#025C3;
entity("trianglelefteq") -> 16#022B4;
entity("triangleq") -> 16#0225C;
entity("triangleright") -> 16#025B9;
entity("trianglerighteq") -> 16#022B5;
entity("tridot") -> 16#025EC;
entity("trie") -> 16#0225C;
entity("triminus") -> 16#02A3A;
entity("triplus") -> 16#02A39;
entity("trisb") -> 16#029CD;
entity("tritime") -> 16#02A3B;
entity("trpezium") -> 16#023E2;
entity("tscr") -> 16#1D4C9;
entity("tscy") -> 16#00446;
entity("tshcy") -> 16#0045B;
entity("tstrok") -> 16#00167;
entity("twixt") -> 16#0226C;
entity("twoheadleftarrow") -> 16#0219E;
entity("twoheadrightarrow") -> 16#021A0;
entity("uArr") -> 16#021D1;
entity("uHar") -> 16#02963;
entity("uacute") -> 16#000FA;
entity("uarr") -> 16#02191;
entity("ubrcy") -> 16#0045E;
entity("ubreve") -> 16#0016D;
entity("ucirc") -> 16#000FB;
entity("ucy") -> 16#00443;
entity("udarr") -> 16#021C5;
entity("udblac") -> 16#00171;
entity("udhar") -> 16#0296E;
entity("ufisht") -> 16#0297E;
entity("ufr") -> 16#1D532;
entity("ugrave") -> 16#000F9;
entity("uharl") -> 16#021BF;
entity("uharr") -> 16#021BE;
entity("uhblk") -> 16#02580;
entity("ulcorn") -> 16#0231C;
entity("ulcorner") -> 16#0231C;
entity("ulcrop") -> 16#0230F;
entity("ultri") -> 16#025F8;
entity("umacr") -> 16#0016B;
entity("uml") -> 16#000A8;
entity("uogon") -> 16#00173;
entity("uopf") -> 16#1D566;
entity("uparrow") -> 16#02191;
entity("updownarrow") -> 16#02195;
entity("upharpoonleft") -> 16#021BF;
entity("upharpoonright") -> 16#021BE;
entity("uplus") -> 16#0228E;
entity("upsi") -> 16#003C5;
entity("upsih") -> 16#003D2;
entity("upsilon") -> 16#003C5;
entity("upuparrows") -> 16#021C8;
entity("urcorn") -> 16#0231D;
entity("urcorner") -> 16#0231D;
entity("urcrop") -> 16#0230E;
entity("uring") -> 16#0016F;
entity("urtri") -> 16#025F9;
entity("uscr") -> 16#1D4CA;
entity("utdot") -> 16#022F0;
entity("utilde") -> 16#00169;
entity("utri") -> 16#025B5;
entity("utrif") -> 16#025B4;
entity("uuarr") -> 16#021C8;
entity("uuml") -> 16#000FC;
entity("uwangle") -> 16#029A7;
entity("vArr") -> 16#021D5;
entity("vBar") -> 16#02AE8;
entity("vBarv") -> 16#02AE9;
entity("vDash") -> 16#022A8;
entity("vangrt") -> 16#0299C;
entity("varepsilon") -> 16#003F5;
entity("varkappa") -> 16#003F0;
entity("varnothing") -> 16#02205;
entity("varphi") -> 16#003D5;
entity("varpi") -> 16#003D6;
entity("varpropto") -> 16#0221D;
entity("varr") -> 16#02195;
entity("varrho") -> 16#003F1;
entity("varsigma") -> 16#003C2;
entity("varsubsetneq") -> [16#0228A, 16#0FE00];
entity("varsubsetneqq") -> [16#02ACB, 16#0FE00];
entity("varsupsetneq") -> [16#0228B, 16#0FE00];
entity("varsupsetneqq") -> [16#02ACC, 16#0FE00];
entity("vartheta") -> 16#003D1;
entity("vartriangleleft") -> 16#022B2;
entity("vartriangleright") -> 16#022B3;
entity("vcy") -> 16#00432;
entity("vdash") -> 16#022A2;
entity("vee") -> 16#02228;
entity("veebar") -> 16#022BB;
entity("veeeq") -> 16#0225A;
entity("vellip") -> 16#022EE;
entity("verbar") -> 16#0007C;
entity("vert") -> 16#0007C;
entity("vfr") -> 16#1D533;
entity("vltri") -> 16#022B2;
entity("vnsub") -> [16#02282, 16#020D2];
entity("vnsup") -> [16#02283, 16#020D2];
entity("vopf") -> 16#1D567;
entity("vprop") -> 16#0221D;
entity("vrtri") -> 16#022B3;
entity("vscr") -> 16#1D4CB;
entity("vsubnE") -> [16#02ACB, 16#0FE00];
entity("vsubne") -> [16#0228A, 16#0FE00];
entity("vsupnE") -> [16#02ACC, 16#0FE00];
entity("vsupne") -> [16#0228B, 16#0FE00];
entity("vzigzag") -> 16#0299A;
entity("wcirc") -> 16#00175;
entity("wedbar") -> 16#02A5F;
entity("wedge") -> 16#02227;
entity("wedgeq") -> 16#02259;
entity("weierp") -> 16#02118;
entity("wfr") -> 16#1D534;
entity("wopf") -> 16#1D568;
entity("wp") -> 16#02118;
entity("wr") -> 16#02240;
entity("wreath") -> 16#02240;
entity("wscr") -> 16#1D4CC;
entity("xcap") -> 16#022C2;
entity("xcirc") -> 16#025EF;
entity("xcup") -> 16#022C3;
entity("xdtri") -> 16#025BD;
entity("xfr") -> 16#1D535;
entity("xhArr") -> 16#027FA;
entity("xharr") -> 16#027F7;
entity("xi") -> 16#003BE;
entity("xlArr") -> 16#027F8;
entity("xlarr") -> 16#027F5;
entity("xmap") -> 16#027FC;
entity("xnis") -> 16#022FB;
entity("xodot") -> 16#02A00;
entity("xopf") -> 16#1D569;
entity("xoplus") -> 16#02A01;
entity("xotime") -> 16#02A02;
entity("xrArr") -> 16#027F9;
entity("xrarr") -> 16#027F6;
entity("xscr") -> 16#1D4CD;
entity("xsqcup") -> 16#02A06;
entity("xuplus") -> 16#02A04;
entity("xutri") -> 16#025B3;
entity("xvee") -> 16#022C1;
entity("xwedge") -> 16#022C0;
entity("yacute") -> 16#000FD;
entity("yacy") -> 16#0044F;
entity("ycirc") -> 16#00177;
entity("ycy") -> 16#0044B;
entity("yen") -> 16#000A5;
entity("yfr") -> 16#1D536;
entity("yicy") -> 16#00457;
entity("yopf") -> 16#1D56A;
entity("yscr") -> 16#1D4CE;
entity("yucy") -> 16#0044E;
entity("yuml") -> 16#000FF;
entity("zacute") -> 16#0017A;
entity("zcaron") -> 16#0017E;
entity("zcy") -> 16#00437;
entity("zdot") -> 16#0017C;
entity("zeetrf") -> 16#02128;
entity("zeta") -> 16#003B6;
entity("zfr") -> 16#1D537;
entity("zhcy") -> 16#00436;
entity("zigrarr") -> 16#021DD;
entity("zopf") -> 16#1D56B;
entity("zscr") -> 16#1D4CF;
entity("zwj") -> 16#0200D;
entity("zwnj") -> 16#0200C;
entity(_) -> undefined.

%%
%% Tests
%%
-ifdef(TEST).
-include_lib("eunit/include/eunit.hrl").

exhaustive_entity_test() ->
    T = mochiweb_cover:clause_lookup_table(?MODULE, entity),
    [?assertEqual(V, entity(K)) || {K, V} <- T].

charref_test() ->
    1234 = charref("#1234"),
    255 = charref("#xfF"),
    255 = charref(<<"#XFf">>),
    38 = charref("amp"),
    38 = charref(<<"amp">>),
    undefined = charref("not_an_entity"),
    undefined = charref("#not_an_entity"),
    undefined = charref("#xnot_an_entity"),
    ok.

-endif.
